//*********************************************************************
// Example2 is a bounded, constrained, mixed integer, non-linear optimization 
// problem. 
// 
// The goal of the optimization is to minimize the following expression subject to 
// the constraints listed below: 
//  -x1^2 + (2x2 - x3)^3 + (3x3 - 4x4)*(x6 - 5x5) - (x6 - x7)^2
//
//	Subject to these constraints:
//   2x1 -  x2 -  x3 +  x4 - 2x5 +  x6 -  x7  =  40
//   2x1 + 2x2 - 3x3 +  x4 + 2x5 + 2x6 +  x7 >=  60
//   6x1 + 2x2 - 3x3 + 4x4 +  x5 + 2x6       <= 120
//   2x1 +  x2 +  x3 + 2x4 +  x5 -  x6       >=  20
//
//		Varibles have the following bounds
//    0   <= x1 <= 10
//   -5   <= x2 <= 20
//    2.5 <= x3 <= 14.3
//		1   <= x4 <= 15
//		0.5 <= x5 <= 23.4
//		0   <= x6 <= 30
//		0   <= x7 <= 28
//
// 
//   Decision variables x1 and x6 are integer variables with step size of 1
//   Decision variable x4 is an integer variable with step size of 3 
//   Decision variables x2, x3, x5 and x7 are continuous variables
//
// The OptQuest.dll distributed with this example is a demo
// version with the following restrictions:
//		- Maximum variables = 7	
//		- Maximum constraints = 4
//		- Maximum requirements =  4
//		- Maximum iterations = 500
//

//*********************************************************************
#include "ocl.h"
#include <stdio.h>
#include <stdlib.h>
#include <math.h>

double evaluate(double *);

int main(int argc, char **argv)
{
	double ObjVal, DecVar[8];
	long   i, TotalIter, Example, nsol, status;

	if (argc != 2) 
	{
		TotalIter = 200;
	}
	else
		TotalIter = atoi(argv[1]);

	if (TotalIter > 500)
	{
		printf("Warning:  The Demo version is limited to 500 iterations\n");
		TotalIter = 500;
	}

	/* Allocating memory (setting up the problem) */

	Example = OCLSetup(7, 0, 4, 0, "MIN", 999999999);
	if (Example < 0) 
	{
		printf("OCLSetup error code %d\n", Example);
		exit(1);
	}

	/* Defining variables */

	OCLDefineVar(Example, 1,  0.0, OCLGetNull(Example), 10.0, "DIS", 1);
	OCLDefineVar(Example, 2, -5.0, OCLGetNull(Example), 20.0, "CON", 1);
	OCLDefineVar(Example, 3,  2.5, OCLGetNull(Example), 14.3, "CON", 1);
	OCLDefineVar(Example, 4,  1.0, OCLGetNull(Example), 15.0, "DIS", 3);
	OCLDefineVar(Example, 5,  0.5, OCLGetNull(Example), 23.4, "CON", 1);
	OCLDefineVar(Example, 6,  0.0, OCLGetNull(Example), 30.0, "DIS", 1);
	OCLDefineVar(Example, 7,  0.0, OCLGetNull(Example), 28.0, "CON", 1);
	
	/* Defining constraints */

   /* 2x1 -  x2 -  x3 +  x4 - 2x5 +  x6 -  x7 = 40 */
	OCLConsCoeff(Example, 1, 1,  2);
	OCLConsCoeff(Example, 1, 2, -1);
	OCLConsCoeff(Example, 1, 3, -1);
	OCLConsCoeff(Example, 1, 4,  1);
	OCLConsCoeff(Example, 1, 5, -2);
	OCLConsCoeff(Example, 1, 6,  1);
	OCLConsCoeff(Example, 1, 7, -1);
	OCLConsType (Example, 1, OCLEQ);
	OCLConsRhs  (Example, 1, 40);

	/* 2x1 + 2x2 - 3x3 +  x4 + 2x5 + 2x6 +  x7 >= 60 */
	OCLConsCoeff(Example, 2, 1,  2);
	OCLConsCoeff(Example, 2, 2,  2);
	OCLConsCoeff(Example, 2, 3, -3);
	OCLConsCoeff(Example, 2, 4,  1);
	OCLConsCoeff(Example, 2, 5,  2);
	OCLConsCoeff(Example, 2, 6,  2);
	OCLConsCoeff(Example, 2, 7,  1);
	OCLConsType (Example, 2, OCLGE);
	OCLConsRhs  (Example, 2, 60);

	/* 6x1 + 2x2 - 3x3 + 4x4 +  x5 + 2x6 <= 120 */
	OCLConsCoeff(Example, 3, 1,  6);
	OCLConsCoeff(Example, 3, 2,  2);
	OCLConsCoeff(Example, 3, 3, -3);
	OCLConsCoeff(Example, 3, 4,  4);
	OCLConsCoeff(Example, 3, 5,  1);
	OCLConsCoeff(Example, 3, 6,  2);
	OCLConsType (Example, 3, OCLLE);
	OCLConsRhs  (Example, 3, 120);

	/* 2x1 +  x2 +  x3 + 2x4 +  x5 -  x6 >= 20 */
	OCLConsCoeff(Example, 4, 1,  2);
	OCLConsCoeff(Example, 4, 2,  1);
	OCLConsCoeff(Example, 4, 3,  1);
	OCLConsCoeff(Example, 4, 4,  2);
	OCLConsCoeff(Example, 4, 5,  1);
	OCLConsCoeff(Example, 4, 6, -1);
	OCLConsType (Example, 4, OCLGE);
	OCLConsRhs  (Example, 4, 20);

	/* Generate and evaluate TotalIter solutions */
	for (i = 1; i <= TotalIter; i++)
	{
		nsol = OCLGetSolution(Example, DecVar);
		if (nsol < 0) 
		{
			printf("OCLGetSolution error code %d\n", nsol);
			exit(1);
		}
		ObjVal = evaluate(DecVar);
		status = OCLPutSolution(Example, nsol, &ObjVal, (double*)OCLGetNull(Example));
		if (status < 0) 
		{
			printf("OCLPutSolution error code %d\n", status);
			exit(1);
		}
		if (!(i%10)) 
		{
			status = OCLGetBest(Example, DecVar, &ObjVal);
			if (status < 0) 
			{
				printf("OCLGetBest error code %d\n", status);
				exit(1);
			}
			printf("Best solution after %5d iterations is %15.6f\n",i,ObjVal);
		}
	}

	/* Display the best solution found */
	status = OCLGetBest(Example, DecVar, &ObjVal);
	if (status < 0) 
	{
		printf("OCLGetBest error code %d\n", status);
		exit(1);
	}
	for (i=1; i<=7; i++) printf("x[%2d] = %6.2f\n",i, DecVar[i]);

	printf("\nPress any key to end ...\n");
	getchar();


	/* Free memory */	
	status = OCLGoodBye(Example);
	if (status < 0) 
	{
		printf("OCLGoodBye error code %d\n", status);
		exit(1);
	}
	return 0;
}

/* Evaluation function */

double evaluate(double *x)
{
	double result;

   result = -pow(x[1],2) + pow(2*x[2]-x[3],3) + (3*x[3]-4*x[4])*(x[6]-5*x[5]) - 
	       pow(x[6]-x[7],2);

	return(result);
}